<?php
// Get currency from settings
$currency = 'KES'; // Default currency
try {
    $stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'currency'");
    $stmt->execute();
    $currency = $stmt->fetchColumn() ?: 'KES';
} catch (PDOException $e) {
    // Handle error silently
}

// Get notifications
$notifications = [];
try {
    // Low stock items
    $stmt = $pdo->prepare("SELECT name, stock_quantity FROM products WHERE stock_quantity <= 5 ORDER BY stock_quantity ASC");
    $stmt->execute();
    $lowStockItems = $stmt->fetchAll();
    foreach ($lowStockItems as $item) {
        $notifications[] = [
            'type' => 'stock',
            'message' => "Low stock alert: {$item['name']} ({$item['stock_quantity']} remaining)",
            'icon' => 'fas fa-box',
            'color' => 'text-warning'
        ];
    }

    // Credit due customers
    $stmt = $pdo->prepare("
        SELECT c.name, COALESCE(SUM(ct.amount), 0) as amount_due 
        FROM customers c 
        LEFT JOIN credit_transactions ct ON c.id = ct.customer_id 
        WHERE ct.type = 'credit' AND ct.status = 'pending' 
        GROUP BY c.id 
        HAVING amount_due > 0 
        ORDER BY amount_due DESC
    ");
    $stmt->execute();
    $creditDueCustomers = $stmt->fetchAll();
    foreach ($creditDueCustomers as $customer) {
        $notifications[] = [
            'type' => 'credit',
            'message' => "Credit due: {$customer['name']} - " . number_format($customer['amount_due'], 2),
            'icon' => 'fas fa-credit-card',
            'color' => 'text-danger'
        ];
    }
} catch (PDOException $e) {
    // Handle error silently
}

// Get current user info
$currentUser = null;
if (isset($_SESSION['user_id'])) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$_SESSION['user_id']]);
        $currentUser = $stmt->fetch();
    } catch (PDOException $e) {
        // Handle error silently
    }
}

// Fetch store name from settings
$storeName = 'POS System';
try {
    $stmt = $pdo->prepare("SELECT setting_value FROM store_settings WHERE setting_key = 'store_name'");
    $stmt->execute();
    $storeName = $stmt->fetchColumn() ?: 'POS System';
} catch (PDOException $e) {}

// Get low stock products
$stmt = $pdo->prepare("
    SELECT p.id, p.name, p.quantity, p.alert_quantity, c.name as category_name
    FROM products p
    LEFT JOIN categories c ON p.category_id = c.id
    WHERE p.quantity <= p.alert_quantity
    ORDER BY p.quantity ASC
    LIMIT 5
");
$stmt->execute();
$lowStockProducts = $stmt->fetchAll();

// Get overdue credit customers
$stmt = $pdo->prepare("
    SELECT 
        c.id, 
        c.name,
        c.credit_limit,
        MAX(ct.created_at) as last_payment_date,
        SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END) as total_credit_used,
        SUM(CASE WHEN ct.type = 'payment' THEN ct.amount ELSE 0 END) as total_payments,
        DATEDIFF(CURRENT_DATE, MAX(ct.created_at)) as days_overdue
    FROM customers c
    LEFT JOIN credit_transactions ct ON c.id = ct.customer_id
    WHERE c.credit_limit > 0
    GROUP BY c.id, c.name, c.credit_limit
    HAVING (SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END) - 
            SUM(CASE WHEN ct.type = 'payment' THEN ct.amount ELSE 0 END)) > 0
    ORDER BY (SUM(CASE WHEN ct.type = 'credit' THEN ct.amount ELSE 0 END) - 
              SUM(CASE WHEN ct.type = 'payment' THEN ct.amount ELSE 0 END)) DESC
    LIMIT 5
");
$stmt->execute();
$overdueCustomers = $stmt->fetchAll();

// Calculate total notifications
$totalNotifications = count($lowStockProducts) + count($overdueCustomers);

$userAvatar = !empty($currentUser['profile_picture'])
    ? '../uploads/profiles/' . htmlspecialchars($currentUser['profile_picture'])
    : '../assets/img/default-profile.png';
?>
<style>
/* Global Responsive Styles */
:root {
    --sidebar-width: 320px;
    --topbar-height: 70px;
    --content-padding: 1.5rem;
}

body {
    min-height: 100vh;
    background: #f8fafc;
    overflow-x: hidden;
}

.main-content {
    margin-left: var(--sidebar-width);
    padding: calc(var(--topbar-height) + var(--content-padding)) var(--content-padding) var(--content-padding);
    min-height: 100vh;
    transition: margin-left 0.3s ease;
}

@media (max-width: 991px) {
    :root {
        --sidebar-width: 0px;
        --content-padding: 1rem;
    }
    
    .main-content {
        margin-left: 0;
        padding-top: calc(var(--topbar-height) + 1rem);
    }
}

/* Responsive Tables */
.table-responsive {
    overflow-x: auto;
    -webkit-overflow-scrolling: touch;
    margin-bottom: 1rem;
}

@media (max-width: 768px) {
    .table-responsive {
        margin: 0 -1rem;
        padding: 0 1rem;
        width: calc(100% + 2rem);
    }
}

/* Responsive Cards */
.card {
    border-radius: 16px;
    border: none;
    box-shadow: 0 4px 24px rgba(37,99,235,0.08);
    transition: transform 0.2s ease, box-shadow 0.2s ease;
}

.card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 32px rgba(37,99,235,0.12);
}

@media (max-width: 576px) {
    .card {
        border-radius: 12px;
    }
}

/* Responsive Grid */
.row {
    margin-left: -0.75rem;
    margin-right: -0.75rem;
}

@media (max-width: 768px) {
    .row {
        margin-left: -0.5rem;
        margin-right: -0.5rem;
    }
}

/* Responsive Forms */
.form-control {
    border-radius: 10px;
    border: 1px solid #e2e8f0;
    padding: 0.6rem 1rem;
    transition: all 0.2s ease;
}

.form-control:focus {
    border-color: #2563eb;
    box-shadow: 0 0 0 3px rgba(37,99,235,0.1);
}

@media (max-width: 576px) {
    .form-control {
        font-size: 0.95rem;
        padding: 0.5rem 0.8rem;
    }
}

/* Responsive Buttons */
.btn {
    border-radius: 10px;
    padding: 0.6rem 1.2rem;
    font-weight: 500;
    transition: all 0.2s ease;
}

@media (max-width: 576px) {
    .btn {
        padding: 0.5rem 1rem;
        font-size: 0.95rem;
    }
}

/* Responsive Images */
img {
    max-width: 100%;
    height: auto;
}

/* Responsive Typography */
@media (max-width: 576px) {
    h1 { font-size: 1.8rem; }
    h2 { font-size: 1.5rem; }
    h3 { font-size: 1.3rem; }
    h4 { font-size: 1.1rem; }
    h5 { font-size: 1rem; }
    h6 { font-size: 0.9rem; }
}

/* Responsive Spacing */
@media (max-width: 768px) {
    .container {
        padding-left: 1rem;
        padding-right: 1rem;
    }
    
    .py-4 { padding-top: 1.5rem !important; padding-bottom: 1.5rem !important; }
    .my-4 { margin-top: 1.5rem !important; margin-bottom: 1.5rem !important; }
}

/* Responsive Utilities */
@media (max-width: 576px) {
    .d-sm-none { display: none !important; }
    .d-sm-block { display: block !important; }
    .text-sm-center { text-align: center !important; }
    .mt-sm-2 { margin-top: 0.5rem !important; }
    .mb-sm-2 { margin-bottom: 0.5rem !important; }
}

.topbar-modern {
    background: rgba(255,255,255,0.85);
    backdrop-filter: blur(8px) saturate(180%);
    margin-bottom: 1.5rem;
    padding: 0.7rem 1rem;
    display: flex;
    align-items: center;
    justify-content: space-between;
    box-shadow: 0 4px 24px 0 rgba(37,99,235,0.08);
    position: fixed;
    top: 0;
    left: 320px;
    right: 0;
    z-index: 100;
    width: calc(100% - 320px);
    border-bottom: 1px solid rgba(37,99,235,0.08);
    border-radius: 24px 24px 24px 24px !important;
}
.topbar-modern .left {
    display: flex;
    align-items: center;
    gap: 1.2rem;
}
.topbar-modern .menu-btn {
    display: none;
    background: none;
    border: none;
    font-size: 1.6rem;
    color: #2563eb;
    margin-right: 1rem;
    cursor: pointer;
}
@media (max-width: 991px) {
    .topbar-modern {
        left: 0;
        width: 100%;
        padding: 0.7rem 1rem;
        border-radius: 0 !important;
    }
    
    .topbar-modern .search {
        display: none;
    }
    
    .topbar-modern .right {
        gap: 0.8rem;
    }
    
    .topbar-modern .icon-btn {
        font-size: 1.1rem;
    }
}
@media (max-width: 576px) {
    .topbar-modern {
        padding: 0.5rem 0.8rem;
    }
    
    .topbar-modern .logo {
        font-size: 1rem;
    }
    
    .topbar-modern .right {
        gap: 0.5rem;
    }
}
.topbar-modern .logo {
    font-size: 1.2rem;
    font-weight: 800;
    color: #2563eb;
    letter-spacing: 1px;
    text-shadow: 0 2px 8px rgba(37,99,235,0.08);
}
.topbar-modern .search {
    flex: 1;
    display: flex;
    justify-content: center;
}
.topbar-modern .search-bar {
    width: 320px;
    position: relative;
    transition: box-shadow 0.2s;
}
.topbar-modern .search-bar input {
    width: 100%;
    border-radius: 10px;
    border: 1px solid #e2e8f0;
    padding: 0.6rem 1.2rem 0.6rem 2.5rem;
    font-size: 1rem;
    background: rgba(255,255,255,0.95);
    box-shadow: 0 2px 8px rgba(37,99,235,0.04);
    transition: border 0.2s, box-shadow 0.2s;
}
.topbar-modern .search-bar input:focus {
    border: 1.5px solid #2563eb;
    box-shadow: 0 4px 16px rgba(37,99,235,0.10);
    outline: none;
}
.topbar-modern .search-bar .fa-search {
    position: absolute;
    left: 0.9rem;
    top: 50%;
    transform: translateY(-50%);
    color: #2563eb;
    font-size: 1.1rem;
}
.topbar-modern .right {
    display: flex;
    align-items: center;
    gap: 1.2rem;
}
.topbar-modern .icon-btn {
    background: none;
    border: none;
    position: relative;
    font-size: 1.3rem;
    color: #2563eb;
    cursor: pointer;
    transition: color 0.2s;
}
.topbar-modern .icon-btn:hover {
    color: #1e40af;
}
.topbar-modern .notification-badge {
    position: absolute;
    top: -6px;
    right: -6px;
    background: #ef4444;
    color: #fff;
    border-radius: 50%;
    width: 18px;
    height: 18px;
    font-size: 0.75rem;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    box-shadow: 0 2px 8px rgba(239,68,68,0.12);
}
.topbar-modern .avatar {
    width: 32px;
    height: 32px;
    border-radius: 50%;
    object-fit: cover;
    border: 1px solid #2563eb;
    background: #fff;
    cursor: pointer;
    box-shadow: 0 2px 8px rgba(37,99,235,0.08);
}
.topbar-modern .dropdown-menu {
    min-width: 260px;
    border: none;
    border-radius: 12px;
    box-shadow: 0 4px 24px rgba(37,99,235,0.10);
    padding: 0.5rem 0;
    margin-top: 0.7rem;
    background: #fff;
    z-index: 200;
}
.topbar-modern .dropdown-item {
    padding: 0.7rem 1.2rem;
    font-size: 1rem;
    color: #222;
    border-radius: 8px;
    transition: background 0.15s;
    display: flex;
    align-items: center;
    gap: 0.7rem;
}
.topbar-modern .dropdown-item:hover {
    background: #f1f5f9;
    color: #2563eb;
}
.topbar-modern .dropdown-divider {
    margin: 1.2rem 0;
    border-top: 1px solid #e2e8f0;
}
.topbar-modern .notification-list {
    max-height: 320px;
    overflow-y: auto;
}
.topbar-modern .notification-item {
    display: flex;
    align-items: flex-start;
    gap: 0.7rem;
    padding: 0.6rem 1rem;
    border-radius: 8px;
    transition: background 0.15s;
}
.topbar-modern .notification-item:hover {
    background: #f1f5f9;
}
.topbar-modern .notification-icon {
    font-size: 1.1rem;
    margin-top: 0.2rem;
}
.topbar-modern .notification-message {
    flex: 1;
    font-size: 0.97rem;
    color: #334155;
}

/* Main content wrapper */
.main-content {
    margin-left: 320px;
    padding: 1.5rem;
    min-height: calc(100vh - 70px);
    background: #f8fafc;
}

@media (max-width: 991px) {
    .main-content {
        margin-left: 0;
    }
}

/* Adjust body padding */
body {
    padding-top: 70px;
    padding-left: 0;
    background: #f8fafc;
    border-radius: 24px 24px 24px 24px;
}

/* Adjust sidebar position to account for fixed topbar */
.sidebar-modern {
    top: 0;
}

.topbar-modern .search-suggestions {
    position: absolute;
    top: 110%;
    left: 0;
    right: 0;
    z-index: 10;
    background: #fff;
    border: 1px solid #e2e8f0;
    border-radius: 8px;
    box-shadow: 0 4px 16px rgba(37,99,235,0.08);
    padding: 0.5rem 0;
    display: none;
}
.topbar-modern .search-suggestion-item {
    padding: 0.6rem 1.2rem;
    cursor: pointer;
    color: #222;
    font-size: 1.05rem;
    display: flex;
    align-items: center;
    gap: 0.7rem;
    transition: background 0.15s, color 0.15s;
}
.topbar-modern .search-suggestion-item:hover {
    background: #f1f5f9;
    color: #2563eb;
}
.topbar-modern .search-suggestion-item i {
    font-size: 1.1rem;
    color: #2563eb;
}

.profile-header {
    padding: 1rem;
}

.profile-header .avatar {
    margin-bottom: 0.5rem;
}

.profile-header .fw-bold {
    color: #1a1a1a;
}

.profile-header .text-muted {
    color: #6c757d;
}

@media (max-width: 1200px) {
    .topbar-modern .search-bar {
        width: 280px;
    }
}

@media (max-width: 991px) {
    .topbar-modern .search-bar {
        width: 100%;
        max-width: 320px;
    }
}
</style>
<div class="topbar-modern">
    <div class="left">
        <span class="logo"><?php echo htmlspecialchars($storeName); ?> | POS & CREDIT SALE SYSTEM</span>
    </div>
    <div class="search">
        <div class="search-bar position-relative">
            <i class="fas fa-search"></i>
            <input type="text" placeholder="Search anything... (e.g. settings, reports, products, customers)" id="universalSearch" autocomplete="off">
            <div id="searchSuggestions" class="search-suggestions"></div>
        </div>
    </div>
    <div class="right">
        <div class="dropdown">
            <button class="icon-btn" id="notifDropdownBtn" data-bs-toggle="dropdown" aria-expanded="false">
                <i class="fas fa-bell"></i>
                <?php if ($totalNotifications > 0): ?>
                <span class="notification-badge"><?php echo $totalNotifications; ?></span>
                <?php endif; ?>
            </button>
            <div class="dropdown-menu dropdown-menu-end notification-list" aria-labelledby="notifDropdownBtn">
                <?php if (count($lowStockProducts) > 0): ?>
                    <div class="dropdown-item text-center text-danger">
                        <i class="fas fa-exclamation-triangle"></i> Low Stock Alerts
                    </div>
                    <?php foreach ($lowStockProducts as $product): ?>
                        <div class="notification-item">
                            <span class="notification-icon text-danger"><i class="fas fa-box"></i></span>
                            <span class="notification-message">
                                <div class="small text-gray-500"><?php echo htmlspecialchars($product['category_name'] ?? 'Uncategorized'); ?></div>
                                <div><?php echo htmlspecialchars($product['name']); ?></div>
                                <div class="small text-danger">
                                    Only <?php echo $product['quantity']; ?> items left
                                    (Alert: <?php echo $product['alert_quantity']; ?>)
                                </div>
                            </span>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>

                <?php if (count($overdueCustomers) > 0): ?>
                    <div class="dropdown-item text-center text-warning">
                        <i class="fas fa-clock"></i> Overdue Credit
                    </div>
                    <?php foreach ($overdueCustomers as $customer): 
                        $totalDue = $customer['total_credit_used'] - $customer['total_payments'];
                    ?>
                        <div class="notification-item">
                            <span class="notification-icon text-warning"><i class="fas fa-exclamation-triangle"></i></span>
                            <span class="notification-message">
                                <div class="small text-gray-500"><?php echo $customer['days_overdue']; ?> days overdue</div>
                                <div><?php echo htmlspecialchars($customer['name']); ?></div>
                                <div class="small text-warning">
                                    Due: <?php echo $currency . ' ' . number_format($totalDue, 2); ?>
                                </div>
                            </span>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>

                <?php if ($totalNotifications === 0): ?>
                    <div class="dropdown-item text-center text-muted">
                        <i class="fas fa-check-circle"></i> No new notifications
                    </div>
                <?php endif; ?>

                <div class="dropdown-divider"></div>
                <a class="dropdown-item text-center small text-gray-500" href="notifications.php">
                    <i class="fas fa-list"></i> Show All Notifications
                </a>
            </div>
        </div>
        <a href="settings.php" class="icon-btn" title="Settings">
            <i class="fas fa-cog"></i>
        </a>
        <a href="https://vicotechnologies.co.ke" target="_blank" class="icon-btn" title="Help">
            <i class="far fa-question-circle"></i>
        </a>
        <div class="dropdown">
            <button class="icon-btn d-flex align-items-center gap-2" id="profileDropdownBtn" data-bs-toggle="dropdown" aria-expanded="false">
                <img src="<?php echo $userAvatar; ?>" class="avatar" alt="Profile Picture">
            </button>
            <div class="dropdown-menu dropdown-menu-end profile-dropdown" aria-labelledby="profileDropdownBtn">
                <div class="profile-header">
                    <img src="<?php echo $userAvatar; ?>" 
                         class="avatar mb-2" 
                         style="width: 40px; height: 40px;" 
                         alt="Profile Picture">
                    <div class="fw-bold mb-1"><?php echo htmlspecialchars($currentUser['full_name'] ?? 'User'); ?></div>
                    <div class="text-muted" style="font-size:0.85rem;">
                        <?php echo htmlspecialchars($currentUser['email'] ?? ''); ?>
                    </div>
                </div>
                <div class="dropdown-divider"></div>
                <a class="dropdown-item" href="profile.php"><i class="fas fa-user"></i> My Profile</a>
                <a class="dropdown-item" href="settings.php"><i class="fas fa-cog"></i> Settings</a>
                <div class="dropdown-divider"></div>
                <a class="dropdown-item text-danger" href="../auth/logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
            </div>
        </div>
    </div>
</div>
<script>
// Sidebar toggle for mobile
const sidebarToggle = document.getElementById('sidebarToggle');
if (sidebarToggle) {
    sidebarToggle.addEventListener('click', function() {
        const sidebar = document.getElementById('sidebarModern');
        if (sidebar) sidebar.classList.toggle('show');
    });
}

// Universal search bar logic
const universalSearch = document.getElementById('universalSearch');
const searchSuggestions = document.getElementById('searchSuggestions');
const navLinks = [
    { label: 'Dashboard', icon: 'fas fa-home', url: 'dashboard.php', keywords: ['dashboard', 'home', 'main'] },
    { label: 'POS', icon: 'fas fa-cash-register', url: 'pos.php', keywords: ['pos', 'sale', 'point of sale'] },
    { label: 'Reports', icon: 'fas fa-chart-line', url: 'reports.php', keywords: ['reports', 'report', 'analytics', 'sales report'] },
    { label: 'Products', icon: 'fas fa-box', url: 'products.php', keywords: ['products', 'product', 'stock', 'inventory'] },
    { label: 'Customers', icon: 'fas fa-users', url: 'customers.php', keywords: ['customers', 'customer', 'client', 'clients'] },
    { label: 'Credit', icon: 'fas fa-credit-card', url: 'credit.php', keywords: ['credit', 'credit sales', 'debt'] },
    { label: 'Settings', icon: 'fas fa-cog', url: 'settings.php', keywords: ['settings', 'system', 'configuration', 'preferences'] }
];

universalSearch.addEventListener('input', function() {
    const query = this.value.trim().toLowerCase();
    if (!query) {
        searchSuggestions.style.display = 'none';
        searchSuggestions.innerHTML = '';
        return;
    }
    const matches = navLinks.filter(link =>
        link.label.toLowerCase().includes(query) ||
        link.keywords.some(k => k.includes(query) || query.includes(k))
    );
    if (matches.length === 0) {
        searchSuggestions.style.display = 'none';
        searchSuggestions.innerHTML = '';
        return;
    }
    searchSuggestions.innerHTML = matches.map(link =>
        `<div class="search-suggestion-item" data-url="${link.url}"><i class="${link.icon}"></i> ${link.label}</div>`
    ).join('');
    searchSuggestions.style.display = 'block';
});

searchSuggestions.addEventListener('mousedown', function(e) {
    const item = e.target.closest('.search-suggestion-item');
    if (item && item.dataset.url) {
        window.location.href = item.dataset.url;
    }
});

document.addEventListener('click', function(e) {
    if (!universalSearch.contains(e.target) && !searchSuggestions.contains(e.target)) {
        searchSuggestions.style.display = 'none';
    }
});

universalSearch.addEventListener('keydown', function(e) {
    if (e.key === 'Enter') {
        const first = searchSuggestions.querySelector('.search-suggestion-item');
        if (first) {
            window.location.href = first.dataset.url;
        }
    }
});

function resetNotificationCount() {
    // Reset the notification count in the badge
    const notificationBadge = document.querySelector('#notificationDropdown .badge');
    if (notificationBadge) {
        notificationBadge.textContent = '0';
        notificationBadge.style.display = 'none';
    }
    
    // Update localStorage
    localStorage.setItem('notificationCount', '0');
    
    // Send request to mark notifications as read
    fetch('mark_notifications_read.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            console.log('Notifications marked as read');
        }
    })
    .catch(error => console.error('Error:', error));
}
</script> 